# HSV - Hierarchical Separated Values

HSV is both a text-based file format and a streaming protocol, using ASCII control characters. Unlimited nesting (like JSON). No escaping required. Binary data supported via DLE transparency.

**Spec**: [hsvfile.com](https://hsvfile.com)

## Features

- **No escaping** - Quotes, backslashes, newlines are literal data
- **Streaming** - STX/ETX framing with clear block boundaries
- **Unlimited nesting** - SO/SI for nested structures
- **Binary mode** - DLE transparency for raw bytes
- **Parallel parsing** - Records are independent, trivially parallelizable

## Quick Start

```rust
use hsv::{parse, HsvValue};

// Parse a simple record
let data = "\x02name\x1fAlice\x1eage\x1f30\x03";
let doc = hsv::parse(data);

println!("{:?}", doc.records[0].get("name"));
// Some(String("Alice"))
```

## Control Characters

| Code | Hex  | Symbol | Purpose |
|------|------|--------|---------|
| US   | 0x1F | ␟      | Key-value separator |
| RS   | 0x1E | ␞      | Property separator |
| GS   | 0x1D | ␝      | Array element separator |
| FS   | 0x1C | ␜      | Record separator |
| STX  | 0x02 | ␂      | Start data block |
| ETX  | 0x03 | ␃      | End data block |
| SOH  | 0x01 | ␁      | Start header |
| SO   | 0x0E | ␎      | Start nested structure |
| SI   | 0x0F | ␏      | End nested structure |
| DLE  | 0x10 | ␐      | Binary mode escape |

## Examples

### Basic key-value

```
␂name␟Alice␞age␟30␃
```

```rust
let doc = hsv::parse("\x02name\x1fAlice\x1eage\x1f30\x03");
// records: [{"name": "Alice", "age": "30"}]
```

### Arrays

```
␂tags␟admin␝user␝guest␃
```

```rust
let doc = hsv::parse("\x02tags\x1fadmin\x1duser\x1dguest\x03");
// records: [{"tags": ["admin", "user", "guest"]}]
```

### Nested structures

```
␂user␟␎name␟Alice␞email␟a@b.com␏␃
```

```rust
let doc = hsv::parse("\x02user\x1f\x0ename\x1fAlice\x1eemail\x1fa@b.com\x0f\x03");
// records: [{"user": {"name": "Alice", "email": "a@b.com"}}]
```

### Headers

```
␁hsv␟1.0␞type␟users␂name␟Alice␃
```

```rust
let doc = hsv::parse("\x01hsv\x1f1.0\x1etype\x1fusers\x02name\x1fAlice\x03");
// header: {"hsv": "1.0", "type": "users"}
// records: [{"name": "Alice"}]
```

## Parallel vs Sequential

The default `parse()` function uses Rayon for parallel processing of blocks:

```rust
// Parallel (default) - faster for large inputs
let doc = hsv::parse(large_data);

// Sequential - deterministic ordering, better for small inputs
let doc = hsv::parse_sequential(small_data);
```

## Parallel HTML Parsing

HSV enables parallel parsing of HTML-like structured documents—something research has struggled with for 30 years. See `go-html-parallel/` for a proof-of-concept.

Prior research ([HPar 2013](https://research.csc.ncsu.edu/picture/publications/papers/taco14.pdf), [ZOOMM 2013](https://dl.acm.org/doi/10.1145/2442516.2442543), [Servo 2017](https://servo.org/blog/2017/08/23/gsoc-parsing/)) achieved limited results due to HTML's stateful parsing model. HSV changes the question: instead of "how do we parallelize HTML parsing?" it asks "why use a format that requires sequential parsing?"

```bash
cd go-html-parallel && go test -v
```

## Language Implementations

| Language | Directory | Tests |
|----------|-----------|-------|
| Rust | `rust/` | 15 passing |
| C++ | `cpp/` | 31 passing |
| C | `c/` | 10 passing |
| Go | `go/` | 12 passing |
| Java | `java/` | 10 passing |
| Python | `python/` | 14 passing |
| JavaScript/TypeScript | `js/` | 12 passing |
| PHP | `php/` | 13 passing |
| Perl | `perl/` | 11 passing |
| Common Lisp | `lisp/` | 28 passing |
| Fortran | `fortran/` | 16 passing |
| COBOL | `cobol/` | 6 passing |

### Python Example

```python
import hsv

doc = hsv.parse("\x02name\x1fAlice\x1eage\x1f30\x03")
print(doc['records'])
# [{'name': 'Alice', 'age': '30'}]
```

### Go Example

```go
import "hsv"

doc := hsv.Parse("\x02name\x1fAlice\x1eage\x1f30\x03")
fmt.Println(doc.Records[0]["name"])
// Alice
```

### JavaScript Example

```javascript
const hsv = require('./hsv');

const doc = hsv.parse("\x02name\x1fAlice\x1eage\x1f30\x03");
console.log(doc.records[0].name);
// Alice
```

## License

CC0 1.0 - Public Domain

HSV was formalized as an open standard by Danslav Slavenskoj, [Lingenic LLC](https://lingenic.com), on 2026-01-27, based on traditional ASCII control character semantics (ANSI X3.4-1963).
